using System;
using System.Xml;
using System.Xml.XPath;
using System.Xml.Xsl;
using System.Xml.Schema;
using System.Xml.Serialization;
using System.Diagnostics;
using System.Text;
using System.IO;
using System.Net;
using System.CodeDom;

namespace CSharpRecipes
{
	public class XML
    {
        #region 15.1 Wczytywanie i dostp do danych XML w kolejnoci wyznaczonej w dokumencie
        static void Indent(int level)
        {
            for (int i = 0; i < level; i++)
                Console.Write(" ");
        }

        public static void AccessXml()
        {
            string xmlFragment = "<?xml version='1.0'?>" +
                "<!-- Przykadowy kod XML -->" +
                "<?pi myProcessingInstruction?>" +
                "<Root>" + 
                "<Node1 nodeId='1'>Pierwszy wze</Node1>" +
                "<Node2 nodeId='2'>Drugi wze</Node2>" +
                "<Node3 nodeId='3'>Trzeci wze</Node3>" +
                "</Root>";

            byte[] bytes = Encoding.UTF8.GetBytes(xmlFragment);
            using (MemoryStream memStream = new MemoryStream(bytes))
            {
                XmlReaderSettings settings = new XmlReaderSettings();
                // sprawdzenie, czy w kodzie XML znajduj si niedozwolone znaki
                settings.CheckCharacters = true;

                using (XmlReader reader = XmlReader.Create(memStream, settings))
                {
                    int level = 0;
                    while (reader.Read())
                    {
                        switch (reader.NodeType)
                        {
                            case XmlNodeType.CDATA:
                                Indent(level);
                                Console.WriteLine("CDATA: {0}", reader.Value);
                                break;
                            case XmlNodeType.Comment:
                                Indent(level);
                                Console.WriteLine("KOMENTARZ: {0}", reader.Value);
                                break;
                            case XmlNodeType.DocumentType:
                                Indent(level);
                                Console.WriteLine("TYP DOKUMENTU: {0}={1}",
                                    reader.Name, reader.Value);
                                break;
                            case XmlNodeType.Element:
                                Indent(level);
                                Console.WriteLine("ELEMENT: {0}", reader.Name);
                                level++;
                                while (reader.MoveToNextAttribute())
                                {
                                    Indent(level);
                                    Console.WriteLine("ATRYBUT: {0}='{1}'",
                                        reader.Name, reader.Value);
                                }
                                break;
                            case XmlNodeType.EndElement:
                                level--;
                                break;
                            case XmlNodeType.EntityReference:
                                Indent(level);
                                Console.WriteLine("ENCJA: {0}", reader.Name);
                                break;
                            case XmlNodeType.ProcessingInstruction:
                                Indent(level);
                                Console.WriteLine("INSTRUKCJA: {0}={1}",
                                    reader.Name, reader.Value);
                                break;
                            case XmlNodeType.Text:
                                Indent(level);
                                Console.WriteLine("TEKST: {0}", reader.Value);
                                break;
                            case XmlNodeType.XmlDeclaration:
                                Indent(level);
                                Console.WriteLine("DEKLARACJA: {0}={1}",
                                    reader.Name, reader.Value);
                                break;
                        }
                    }
                }
            }
        }
        #endregion

        #region 15.2 Odczyt dokumentu XML z sieci WWW
        public static void ReadXmlWeb()
        {
            // Przykad wymaga zdefiniowania katalogu wirtualnego
            // wskazujcego plik sample.xml       
            string url = "http://localhost/xml/sample.xml";
            using (XmlReader reader = XmlReader.Create(url))
            {
                while (reader.Read())
                {
                    switch (reader.NodeType)
                    {
                        case XmlNodeType.Element:
                            Console.Write("<{0}>", reader.Name);
                            break;
                    }
                }
            }
        }
        #endregion

        #region 15.3 Wyszukiwanie informacji w dokumencie XML
        public static void QueryXml()
        {
            string xmlFragment = "<?xml version='1.0'?>" +
                "<Clue>" + 
                "<Participant type='Perpetrator'>Professor Plum</Participant>" +
                "<Participant type='Witness'>Colonel Mustard</Participant>" +
                "<Participant type='Witness'>Mrs. White</Participant>" +
                "<Participant type='Witness'>Mrs. Peacock</Participant>" +
                "<Participant type='Witness'>Mr. Green</Participant>" +
                "</Clue>";
            using (StringReader reader = new StringReader(xmlFragment))
            {
                // utworzenie egzemplarza XPathDocument przy uyciu StringReader  
                XPathDocument xpathDoc = new XPathDocument(reader);

                // pobranie nawigatora
                XPathNavigator xpathNav = xpathDoc.CreateNavigator();

                // zapytanie, ktre wyszukuje zamne kobiety
                // bdce wiadkami zbrodni                
                string xpathQuery =
                    "/Clue/Participant[attribute::type='Witness'][contains(text(),'Mrs.')]";
                XPathExpression xpathExpr = xpathNav.Compile(xpathQuery);

                // odczytanie zestawu wzw ze skompilowanego wyraenia
                XPathNodeIterator xpathIter = xpathNav.Select(xpathExpr);

                // zwrcenie odnalezionych wzw (w tym przykadzie: Mrs. White oraz Mrs.Peacock)
                while (xpathIter.MoveNext())
                {
                    Console.WriteLine(xpathIter.Current.Value);
                }
            }
        }
        #endregion

        #region 15.4 Weryfikacja poprawnoci danych XML
        public static void ValidateXml()
        {
            // utworzenie kolekcji schematu XSD na podstawie book.xsd
            XmlReaderSettings settings = new XmlReaderSettings();
            // podpicie uchwytu, aby przechwyci bdy weryfikacji
            settings.ValidationEventHandler += settings_ValidationEventHandler;

            // ustawienie typu weryfikacji na schema (w Beta1 bya to waciwo XsdValidate)
            settings.ValidationType = ValidationType.Schema;

            // dodanie book.xsd
            settings.Schemas.Add(null, XmlReader.Create(@"..\..\Book.xsd"));
            // upewnienie si, e book.xsd zosta dodany
            if (settings.Schemas.Count > 0)
            {
                // otwarcie pliku bookbad.xml
                using (XmlReader reader = XmlReader.Create(@"..\..\BookBad.xml", settings))
                {
                    // wczytanie wszystkich wzw i ich wywietlenie
                    while (reader.Read())
                    {
                        if (reader.NodeType == XmlNodeType.Element)
                        {
                            Console.Write("<{0}", reader.Name);
                            while (reader.MoveToNextAttribute())
                            {
                                Console.Write(" {0}='{1}'", reader.Name,
                                    reader.Value);
                            }
                            Console.Write(">");
                        }
                        else if (reader.NodeType == XmlNodeType.Text)
                        {
                            Console.Write(reader.Value);
                        }
                        else if (reader.NodeType == XmlNodeType.EndElement)
                        {
                            Console.WriteLine("</{0}>", reader.Name);
                        }
                    }
                }
            }
        }

        private static void settings_ValidationEventHandler(object sender, ValidationEventArgs e) 
        {
            Console.WriteLine("Bd weryfikacji poprawnoci: {0}", e.Message);
            Console.WriteLine("Poziom bdu weryfikacji poprawnoci: {0}", e.Severity);
            if (e.Exception != null)
            {
                Console.WriteLine("Numer wiersza bdu weryfikacji poprawnoci: {0}", e.Exception.LineNumber);
                Console.WriteLine("Pozycja w wierszu bdu weryfikacji poprawnoci: {0}", e.Exception.LinePosition);
                Console.WriteLine("rdo bdu weryfikacji poprawnoci: {0}", e.Exception.Source);
                Console.WriteLine("Schemat rdowy bdu weryfikacji poprawnoci: {0}", e.Exception.SourceSchemaObject);
                Console.WriteLine("rdowe URI bdu weryfikacji poprawnoci: {0}", e.Exception.SourceUri);
                Console.WriteLine("Miejsce pochodzenia bdu weryfikacji poprawnoci: {0}", e.Exception.TargetSite);
                Console.WriteLine("Stos dla bdu weryfikacji poprawnoci: {0}", e.Exception.StackTrace);
            }
        }
        #endregion

        #region 15.5 Tworzenie dokumentu XML z poziomu kodu rdowego
        public static void CreateXml()
        {
            XmlWriterSettings settings = new XmlWriterSettings();
            settings.Indent = true;
            using (XmlWriter writer = XmlWriter.Create(Console.Out, settings))
            {
                writer.WriteStartElement("AddressBook");
                writer.WriteStartElement("Contact");
                writer.WriteAttributeString("name", "Kowalski");
                writer.WriteAttributeString("phone", "999-888-0000");
                writer.WriteEndElement();
                writer.WriteStartElement("Contact");
                writer.WriteAttributeString("name", "Grski");
                writer.WriteAttributeString("phone", "666-666-6666");
                writer.WriteEndElement();
                writer.WriteStartElement("Contact");
                writer.WriteAttributeString("name", "Kaczmarek");
                writer.WriteAttributeString("phone", "777-555-3333");
                writer.WriteEndElement();
                writer.WriteEndElement();
            }

            // na pocztek trzeba utworzy XmlDocument
            XmlDocument xmlDoc = new XmlDocument();
            // utworzenie gwnego wza dokumentu
            XmlElement addrBook = xmlDoc.CreateElement("AddressBook");
            xmlDoc.AppendChild(addrBook);
            // utworzenie wpisu dla Kowalskiego
            XmlElement contact = xmlDoc.CreateElement("Contact");
            contact.SetAttribute("name","Kowalski");
            contact.SetAttribute("phone","999-888-0000");
            addrBook.AppendChild(contact);
            // utworzenie wpisu dla Grskiego
            contact = xmlDoc.CreateElement("Contact");
            contact.SetAttribute("name","Grski");
            contact.SetAttribute("phone","666-666-6666");
            addrBook.AppendChild(contact);
            // utworzenie wpisu dla Kaczmarka
            contact = xmlDoc.CreateElement("Contact");
            contact.SetAttribute("name","Kaczmarek");
            contact.SetAttribute("phone","777-555-3333");
            addrBook.AppendChild(contact);

            // wywietlenie XML-a
            Console.WriteLine("Wygenerowany XML:\r\n{0}",addrBook.OuterXml);
            Console.WriteLine();
        }
        #endregion

        #region 15.6 Wykrywanie zmian w dokumencie XML
        public static void DetectXmlChanges()
        {
            string xmlFragment = "<?xml version='1.0'?>" +
                "<!-- Przykadowy dokument XML -->" +
                "<?pi myProcessingInstruction?>" +
                "<Root>" + 
                "<Node1 nodeId='1'>Pierwszy wze</Node1>" +
                "<Node2 nodeId='2'>Drugi wze</Node2>" +
                "<Node3 nodeId='3'>Trzeci wze</Node3>" +
                @"<Node4><![CDATA[<>\&']]></Node4>" +
                "</Root>"; 

            XmlDocument doc = new XmlDocument();
            doc.LoadXml(xmlFragment);

            // utworzenie procedur obsugi zdarze
            doc.NodeChanging += new XmlNodeChangedEventHandler(NodeChangingEvent);
            doc.NodeChanged += new XmlNodeChangedEventHandler(NodeChangedEvent);
            doc.NodeInserting += new XmlNodeChangedEventHandler(NodeInsertingEvent);
            doc.NodeInserted += new XmlNodeChangedEventHandler(NodeInsertedEvent);
            doc.NodeRemoving += new XmlNodeChangedEventHandler(NodeRemovingEvent);
            doc.NodeRemoved += new XmlNodeChangedEventHandler(NodeRemovedEvent);

            // dodanie nowego wza elementu
            XmlElement elem = doc.CreateElement("Node5");
            XmlText text = doc.CreateTextNode("Pity element");
            doc.DocumentElement.AppendChild(elem);
            doc.DocumentElement.LastChild.AppendChild(text);

            // zmiana pierwszego wza
            doc.DocumentElement.FirstChild.InnerText = "1. wze";

            // usunicie czwartego wza
            XmlNodeList nodes = doc.DocumentElement.ChildNodes;
            foreach(XmlNode node in nodes)
            {
                if(node.Name == "Node4")
                {
                    doc.DocumentElement.RemoveChild(node);
                    break;
                }
            }            

            // zwrcenie nowego kodu XML
            Console.WriteLine(doc.OuterXml);
        }

        private static void WriteNodeInfo(string action, XmlNode node)
        {
            if (node.Value != null)
            {
                Console.WriteLine("Element: <{0}> {1} z wartoci {2}", 
                    node.Name,action,node.Value);
            }
            else
                Console.WriteLine("Element: <{0}> {1} z wartoci pust", 
                    node.Name,action);
        }

        public static void NodeChangingEvent(object source, XmlNodeChangedEventArgs e)
        {
            WriteNodeInfo("zmiana",e.Node);
        }

        public static void NodeChangedEvent(object source, XmlNodeChangedEventArgs e)
        {
            WriteNodeInfo("zmieniony",e.Node);
        }

        public static void NodeInsertingEvent(object source, XmlNodeChangedEventArgs e)
        {
            WriteNodeInfo("wstawienie",e.Node);
        }

        public static void NodeInsertedEvent(object source, XmlNodeChangedEventArgs e)
        {
            WriteNodeInfo("wstawiony",e.Node);
        }

        public static void NodeRemovingEvent(object source, XmlNodeChangedEventArgs e)
        {
            WriteNodeInfo("usuwanie",e.Node);
        }

        public static void NodeRemovedEvent(object source, XmlNodeChangedEventArgs e)
        {
            WriteNodeInfo("usunity",e.Node);
        }
        
        #endregion

        #region 15.7 Obsuga niedozwolonych znakw w cigu znakw XML
        public static void HandleInvalidChars()
        {
            // zdefiniowanie cigu z niedozwolonymi znakami
            string invalidChars = @"<>\&'";
            XmlWriterSettings settings = new XmlWriterSettings();
            settings.Indent = true;
            using (XmlWriter writer = XmlWriter.Create(Console.Out, settings))
            {
                writer.WriteStartElement("Root");
                writer.WriteStartElement("InvalidChars1");
                writer.WriteCData(invalidChars);
                writer.WriteEndElement();
                writer.WriteElementString("InvalidChars2", invalidChars);
                writer.WriteEndElement();
            }

            // zdefiniowanie cigu z niedozwolonymi znakami
            invalidChars = @"<>\&'";

            XmlDocument xmlDoc = new XmlDocument();
            // utworzenie gwnego wza dokumentu
            XmlElement root = xmlDoc.CreateElement("Root");
            xmlDoc.AppendChild(root);

            // utworzenie pierwszego wza z niedozwolonymi znakami
            XmlElement invalidElement1 = xmlDoc.CreateElement("InvalidChars1");
            // umieszczenie niedozwolonych znakw w sekcji CDATA i uycie
            // waciwoci InnerXML do przypisania wartoci,
            // poniewa nie zamienia ona znakw niedozwolonych w dozwolone odpowiedniki,
            // a jedynie przekazuje otrzymany tekst            
            invalidElement1.AppendChild(xmlDoc.CreateCDataSection(invalidChars));
            // dodanie elementu do gwnego wza
            root.AppendChild(invalidElement1);

            // utworzenie drugiego wza z niedozwolonymi znakami
            XmlElement invalidElement2 = xmlDoc.CreateElement("InvalidChars2");
            // dodanie niedozwolonych znakw w sposb bezporedni
            // za pomoc waciwoci InnerText, w celu przypisania wartoci;
            // w ten sposb znaki niedozwolone zostan automatycznie zastpione
            // dozwolonymi odpowiednikami            
            invalidElement2.InnerText = invalidChars;
            // dodanie elementu do gwnego wza
            root.AppendChild(invalidElement2);

            Console.WriteLine("Wygenerowany XML z niedozwolonymi znakami:\r\n{0}",xmlDoc.OuterXml);
            Console.WriteLine();
        }
        #endregion

        #region 15.8 Przeksztacanie danych XML
        public static void TransformXml()
        {
            // utworzenie obiektu XmlUrlResolever z danymi domylnymi
            XmlUrlResolver resolver = new XmlUrlResolver();
            resolver.Credentials = System.Net.CredentialCache.DefaultCredentials;

            // przeksztacenie danych z personnel.xml do postaci html
            XslCompiledTransform transform = new XslCompiledTransform();
            XsltSettings settings = new XsltSettings();
            // zablokowanie obydwu ustawie (domylnych) ze wzgldw bezpieczestwa
            settings.EnableDocumentFunction = false;
            settings.EnableScript = false;
            // zaadowanie arkusza stylw
            transform.Load(@"..\..\PersonnelHTML.xsl",settings,resolver);
            // wykonanie przeksztacenia
            transform.Transform(@"..\..\Personnel.xml",@"..\..\Personnel.html");


            // przeksztacenie pliku personnel.xml do postaci danych oddzielonych przecinkami

            // zaadowanie arkusza stylw
            transform.Load(@"..\..\PersonnelCSV.xsl",settings,resolver);
            // wykonanie przeksztacenia
            transform.Transform(@"..\..\Personnel.xml",
                @"..\..\Personnel.csv");
        }
        #endregion

        #region 15.9 Dzielenie dokumentu XML na czci
        public static void ProcessInvoice()
        {
            XmlDocument xmlDoc = new XmlDocument();
            // pobranie faktury z katalogu
            xmlDoc.Load(@"..\..\Invoice.xml");
            // odczytanie wza elementu Invoice
            XmlNode Invoice = xmlDoc.SelectSingleNode("/Invoice");

            // odczytanie atrybutu daty faktury
            XmlAttribute invDate = 
                (XmlAttribute)Invoice.Attributes.GetNamedItem("invoiceDate");
            // odczytanie atrybutu numeru faktury
            XmlAttribute invNum = 
                (XmlAttribute)Invoice.Attributes.GetNamedItem("invoiceNumber");

            // przetworzenie danych finansowych dla dziau finansowego
            WriteInformation(@"..\..\BillingEnvelope.xml",
                            "BillingEnvelope",
                            invDate, invNum, xmlDoc,
                            "/Invoice/billInfo");

            // przetworzenie danych o wysyce dla dziau dystrybucji
            WriteInformation(@"..\..\ShippingEnvelope.xml",
                            "ShippingEnvelope",
                            invDate, invNum, xmlDoc,
                            "/Invoice/shipInfo");

            // przetworzenie informacji o produktach dla magazynu
            WriteInformation(@"..\..\FulfillmentEnvelope.xml",
                            "FulfillmentEnvelope",
                            invDate, invNum, xmlDoc,
                            "/Invoice/Items/item");

            // wysanie danych do usug sieciowych 
        }

        private static void WriteInformation(string path,
                                    string rootNode,
                                    XmlAttribute invDate,
                                    XmlAttribute invNum,
                                    XmlDocument xmlDoc,
                                    string nodePath)
        {
            XmlWriterSettings settings = new XmlWriterSettings();
            settings.Indent = true;
            using (XmlWriter writer = XmlWriter.Create(path, settings))
            {
                writer.WriteStartDocument();
                writer.WriteStartElement(rootNode);
                writer.WriteAttributeString(invDate.Name, invDate.Value);
                writer.WriteAttributeString(invNum.Name, invNum.Value);
                XmlNodeList nodeList = xmlDoc.SelectNodes(nodePath);
                // dodanie informacji finansowych do "koperty"
                foreach (XmlNode node in nodeList)
                {
                    writer.WriteRaw(node.OuterXml);
                }
                writer.WriteEndElement();
                writer.WriteEndDocument();
            }
        }

        #endregion

        #region 15.10 Skadanie dokumentu XML z czci
        public static void ReceiveInvoice()
        {
            XmlDocument invoice = new XmlDocument();
            XmlDocument billing = new XmlDocument();
            XmlDocument shipping = new XmlDocument();
            XmlDocument fulfillment = new XmlDocument();

            // utworzenie gwnego wza faktury
            XmlElement invoiceElement = invoice.CreateElement("Invoice");
            invoice.AppendChild(invoiceElement);
    
            // zaadowanie czci finansowej
            billing.Load(@"..\..\BillingEnvelope.xml");
            // odczytanie atrybutu daty faktury
            XmlAttribute invDate = (XmlAttribute)billing.DocumentElement.Attributes.GetNamedItem("invoiceDate");
            // odczytanie atrybutu numeru faktury
            XmlAttribute invNum = (XmlAttribute)billing.DocumentElement.Attributes.GetNamedItem("invoiceNumber");
            // utworzenie faktury na podstawie posiadanych danych
            invoice.DocumentElement.Attributes.SetNamedItem(invDate.Clone());
            invoice.DocumentElement.Attributes.SetNamedItem(invNum.Clone());
            // dodanie fragmentu billInfo
            XmlNodeList billList = billing.SelectNodes("/BillingEnvelope/billInfo");
            foreach(XmlNode billInfo in billList)
            {
                invoice.DocumentElement.AppendChild(invoice.ImportNode(billInfo,true));
            }

            // zaadowanie danych o wysyce 
            shipping.Load(@"..\..\ShippingEnvelope.xml");
            // dodanie fragmentu shipInfo
            XmlNodeList shipList = shipping.SelectNodes("/ShippingEnvelope/shipInfo");
            foreach(XmlNode shipInfo in shipList)
            {
                invoice.DocumentElement.AppendChild(invoice.ImportNode(shipInfo,true));
            }

            // zaadowanie produktw
            fulfillment.Load(@"..\..\FulfillmentEnvelope.xml");
    
            // utworzenie elementu Items w elemencie Invoice, aby doda do niego produkty
            XmlElement items = invoice.CreateElement("Items");

            // dodanie elementw item do elementu Items
            XmlNodeList itemList = fulfillment.SelectNodes("/FulfillmentEnvelope/item");
            foreach(XmlNode item in itemList)
            {
                items.AppendChild(invoice.ImportNode(item,true));
            }

            // dodanie produktw
            invoice.DocumentElement.AppendChild(items.Clone());

            // wywietlenie faktury w postaci XML
            Console.WriteLine("Faktura:\r\n{0}",invoice.OuterXml);

            // zapisanie odtworzonej faktury
            invoice.Save(@"..\..\ReceivedInvoice.xml");
        }
        #endregion

        #region 15.11 Weryfikacja poprawnoci zmienionego dokumentu XML bez jego ponownego adowania
        // ustawienie pomylnego wyniku pocztkowej weryfikacji poprawnoci
        static bool bValidXml = true;
        public static void TestContinualValidation()
        {
            string xmlFile = @"..\..\Book.xml";
            string xsdFile = @"..\..\Book.xsd";

            // utworzenie zestawu schematu
            XmlSchemaSet schemaSet = new XmlSchemaSet();
            // dodanie nowego schematu z docelow przestrzeni nazw
            // (gdyby schematw byo wicej, mona by doda je wszystkie za jednym razem)
            schemaSet.Add("http://tempuri.org/Book.xsd", XmlReader.Create(xsdFile));

            // zaadowanie pliku xml
            XmlDocument xmlDoc = new XmlDocument();
            // dodanie schematu
            xmlDoc.Schemas = schemaSet;
            // weryfikacja poprawnoci po zaadowaniu
            xmlDoc.Load(xmlFile);
            ValidationEventHandler eventHandler = ValidationEventHandler_15_11;
            xmlDoc.Validate(eventHandler);

            // dodanie nowego wza, ktrego nie ma w schemacie
            // poniewa weryfikacja ju nastpia, nie bdzie informacji zwrotnej...            
            XmlNode newNode = xmlDoc.CreateElement("BogusElement");
            newNode.InnerText = "Razem";
            // dodanie nowego elementu
            xmlDoc.DocumentElement.AppendChild(newNode);
            // weryfikacja poprawnoci dodanego nowego elementu
            xmlDoc.Validate(eventHandler);

            if (bValidXml == true)
            {
                Console.WriteLine("Pomylnie zweryfikowano poprawno zmodyfikowanego dokumentu XML");
            }
            else
            {
                Console.WriteLine("Zmodyfikowany dokument XML nie jest prawidowy");
            }
        }

        private static void ValidationEventHandler_15_11(object sender, ValidationEventArgs e)
        {
            // wywoano t metod, wic dokument jest nieprawidowy
            bValidXml = false;
            Console.WriteLine("Bd weryfikacji poprawnoci: {0}", e.Message);
            Console.WriteLine("Poziom bdu weryfikacji poprawnoci: {0}", e.Severity);
            if (e.Exception != null)
            {
                Console.WriteLine("Numer wiersza bdu weryfikacji poprawnoci: {0}", e.Exception.LineNumber);
                Console.WriteLine("Pozycja w wierszu bdu weryfikacji poprawnoci: {0}", e.Exception.LinePosition);
                Console.WriteLine("rdo bdu weryfikacji poprawnoci: {0}", e.Exception.Source);
                Console.WriteLine("Schemat rdowy bdu weryfikacji poprawnoci: {0}", e.Exception.SourceSchemaObject);
                Console.WriteLine("rdowe URI bdu weryfikacji poprawnoci: {0}", e.Exception.SourceUri);
                Console.WriteLine("Miejsce pochodzenia bdu weryfikacji poprawnoci: {0}", e.Exception.TargetSite);
                Console.WriteLine("Stos dla bdu weryfikacji poprawnoci: {0}", e.Exception.StackTrace);
            }
        }
        #endregion

        #region 15.12 Rozszerzanie przeksztace XSLT
        public static void TestExtendingTransformations()
        {
            string xmlFile = @"..\..\publications.xml";
            string xslt = @"..\..\publications.xsl";

            // utworzenie obiektu XslCompiledTransform i zaadowanie arkusza stylw
            XslCompiledTransform transform = new XslCompiledTransform();
            transform.Load(xslt);
            // zaadowanie pliku xml
            XPathDocument xPathDoc = new XPathDocument(xmlFile);

            // utworzenie argumentw dla arkusza stylw z obiektem rozszerzenia
            XsltArgumentList xslArg = new XsltArgumentList();
            XslExtensionObject xslExt = new XslExtensionObject();
            xslArg.AddExtensionObject("urn:xslext", xslExt);

            // wysanie danych wynikowych na konsol i wykonanie przeksztacenia
            using (XmlWriter writer = XmlWriter.Create(Console.Out))
            {
                transform.Transform(xPathDoc, xslArg, writer);
            }
        }

        // obiekt rozszerzenia wzbogacajcy moliwoci przeksztace
        public class XslExtensionObject
        {
            public XPathNodeIterator GetErrata(XPathNodeIterator nodeChapter)
            {
                try
                {
                    // w tym miejscu mona wykona dodatkowe operacje wyszukiwania
                    // (w XML-u, bazie danych, usudze sieciowej) i odczyta informacje,
                    // ktre zostan dodane do wynikw przeksztacenia
                    string errata = string.Format("<Errata>{0} posiada {1} poprawek</Errata>", nodeChapter.Current.Value, nodeChapter.Current.Value.Length);
                    XmlDocument xDoc = new XmlDocument();
                    xDoc.LoadXml(errata);
                    XPathNavigator xPathNav = xDoc.CreateNavigator();
                    xPathNav.MoveToChild(XPathNodeType.Element);
                    XPathNodeIterator iter = xPathNav.Select(".");
                    return iter;
                }
                catch (Exception e)
                {
                    // obsuga wyjtku, przez ktry nie udao si wykorzysta obiektu
                    // zwrcenie oryginalnego iteratora
                    //Debug.WriteLine(e.ToString());
                    return nodeChapter;
                }
            }
        }

		#endregion

		#region 15.13 Odczytywanie schematu z istniejcych plikw XML
        public static void TestBulkSchema()
        {
            DirectoryInfo di = new DirectoryInfo(@"..\..");
            string dir = di.FullName;
            GenerateSchemaForDirectory(dir);
        }

        public static void GenerateSchemaForDirectory(string dir)
        {
            // upewnienie si, e katalog istnieje
            if (Directory.Exists(dir))
            {
                // odczytanie plikw znajdujcych si w katalogu
                string[] files = Directory.GetFiles(dir, "*.xml");
                foreach (string file in files)
                {
                    // utworzenie obiektu czytajcego plik
                    using (XmlReader reader = XmlReader.Create(file))
                    {
                        XmlSchemaSet schemaSet = new XmlSchemaSet();
                        XmlSchemaInference schemaInference =
                                        new XmlSchemaInference();

                        // odczytanie schematu
                        schemaSet = schemaInference.InferSchema(reader);

                        string schemaPath = "";
                        foreach (XmlSchema schema in schemaSet.Schemas())
                        {
                            // utworzenie cieki do pliku schematu
                            schemaPath = Path.GetDirectoryName(file) + @"\" +
                                            Path.GetFileNameWithoutExtension(file) + ".xsd";
                            using (FileStream fs =
                                new FileStream(schemaPath, FileMode.OpenOrCreate))
                            {
                                schema.Write(fs);
                            }
                        }
                    }
                }
            }
        }
		#endregion

		#region 15.14 Przekazywanie parametrw do transformacji XSLT
        public static void TestXSLTParams()
        {
            XsltArgumentList args = new XsltArgumentList();
            args.AddParam("storeTitle", "", "Bohaterowie Komiksw");
            args.AddParam("pageDate", "", DateTime.Now.ToString("F"));

            // utworzenie obiektu XmlUrlResolver z danymi domylnymi
            XmlUrlResolver resolver = new XmlUrlResolver();
            resolver.Credentials = System.Net.CredentialCache.DefaultCredentials;

            XslCompiledTransform transform = new XslCompiledTransform();
            // zaadowanie arkusza stylw
            transform.Load(@"..\..\ParameterExample.xslt", XsltSettings.Default, resolver);
            // wykonanie transformacji
            FileStream fs = null;
            using (fs =
                new FileStream(@"..\..\ParameterExample.htm",
                                FileMode.OpenOrCreate, FileAccess.Write))
            {
                transform.Transform(@"..\..\ParameterExample.xml", args, fs);
            }

            // zmiana parametrw i ponowne przetworzenie
            args = new XsltArgumentList();
            args.AddParam("storeTitle", "", "Niezwyke przygody");
            args.AddParam("pageDate", "", DateTime.Now.ToString("D"));
            using (fs = new FileStream(@"..\..\ParameterExample2.htm",
                FileMode.OpenOrCreate, FileAccess.Write))
            {
                transform.Transform(@"..\..\ParameterExample.xml", args, fs);
            }
        }
		#endregion
		

	}
}
